<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Campaign Setup Wizard</title>
    <style>
        :root {
            --primary-color: #1a237e;
            --secondary-color: #e53935;
            --accent-color: #3949ab;
            --light-color: #f5f5f5;
            --success-color: #4caf50;
            --warning-color: #ff9800;
            --error-color: #f44336;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        body {
            background-color: #f8f9fa;
            color: #333;
            line-height: 1.6;
            padding: 20px;
        }
        
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 12px;
            box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
            padding: 30px;
            position: relative;
            overflow: hidden;
        }
        
        h1, h2, h3, h4 {
            color: var(--primary-color);
            margin-bottom: 15px;
        }
        
        h1 {
            text-align: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 2px solid #eee;
        }
        
        p {
            margin-bottom: 20px;
        }
        
        .budget-container, .campaign-details {
            background: #f9f9f9;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 25px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
        }
        
        .price-display {
            font-size: 2.5rem;
            font-weight: bold;
            text-align: center;
            color: var(--secondary-color);
            margin: 15px 0;
            padding: 10px;
            background: white;
            border-radius: 8px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }
        
        .slider-wrapper {
            margin: 25px 0;
        }
        
        .animated-hand-text-container {
            text-align: center;
            margin-bottom: 15px;
        }
        
        .animated-text {
            font-weight: bold;
            color: var(--accent-color);
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% { opacity: 0.7; }
            50% { opacity: 1; }
            100% { opacity: 0.7; }
        }
        
        .hand-icon {
            font-size: 1.2rem;
            display: inline-block;
            animation: bounce 1s infinite alternate;
        }
        
        .hand-left {
            animation-delay: 0.2s;
        }
        
        .hand-right {
            animation-delay: 0.4s;
        }
        
        @keyframes bounce {
            from { transform: translateY(0); }
            to { transform: translateY(-5px); }
        }
        
        .budget-slider {
            width: 100%;
            height: 15px;
            -webkit-appearance: none;
            appearance: none;
            background: linear-gradient(to right, #4caf50, #ffeb3b, #f44336);
            outline: none;
            border-radius: 10px;
            opacity: 0.7;
            transition: opacity .2s;
        }
        
        .budget-slider:hover {
            opacity: 1;
        }
        
        .budget-slider::-webkit-slider-thumb {
            -webkit-appearance: none;
            appearance: none;
            width: 25px;
            height: 25px;
            border-radius: 50%;
            background: var(--primary-color);
            cursor: pointer;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.2);
        }
        
        .campaign-details ul {
            list-style-type: none;
            padding: 15px;
            background: white;
            border-radius: 8px;
        }
        
        .campaign-details li {
            padding: 8px 0;
            border-bottom: 1px solid #eee;
        }
        
        .campaign-details li:last-child {
            border-bottom: none;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #555;
        }
        
        input[type="text"], textarea, select {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 6px;
            font-size: 1rem;
            transition: border-color 0.3s;
        }
        
        input[type="text"]:focus, textarea:focus, select:focus {
            border-color: var(--accent-color);
            outline: none;
            box-shadow: 0 0 0 2px rgba(57, 73, 171, 0.2);
        }
        
        textarea {
            min-height: 120px;
            resize: vertical;
        }
        
        .btn-container {
            display: flex;
            justify-content: center;
            gap: 15px;
            margin-top: 30px;
        }
        
        button {
            padding: 12px 25px;
            border: none;
            border-radius: 6px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            color: white;
        }
        
        .btn-primary:hover {
            background-color: var(--accent-color);
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .btn-secondary {
            background-color: #e0e0e0;
            color: #333;
        }
        
        .btn-secondary:hover {
            background-color: #d0d0d0;
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
        }
        
        .tab-container {
            display: flex;
            justify-content: center;
            margin-bottom: 25px;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .tab-button {
            padding: 10px 15px;
            background: #e9ecef;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .tab-button.active, .tab-button:hover {
            background: var(--primary-color);
            color: white;
        }
        
        .coupon-box {
            display: flex;
            align-items: center;
            margin-bottom: 10px;
        }
        
        #coupon-code {
            flex: 1;
            margin-right: 10px;
        }
        
        .validation-icon {
            margin-left: 10px;
            font-size: 1.2rem;
        }
        
        .validation-msg {
            margin-left: 10px;
            padding: 5px 10px;
            border-radius: 4px;
            font-size: 0.9rem;
        }
        
        .validation-success {
            background-color: #e8f5e9;
            color: var(--success-color);
        }
        
        .validation-error {
            background-color: #ffebee;
            color: var(--error-color);
        }
        
        .validation-warning {
            background-color: #fff8e1;
            color: var(--warning-color);
        }
        
        .coupon-btn {
            padding: 8px 15px;
            background: var(--accent-color);
            color: white;
            border: none;
            border-radius: 4px;
            cursor: pointer;
        }
        
        .custom-dropdown {
            position: relative;
            width: 100%;
        }
        
        .dropdown-select {
            width: 100%;
            padding: 12px 15px;
            border: 1px solid #ddd;
            border-radius: 6px;
            background: white;
            font-size: 1rem;
            cursor: pointer;
        }
        
        .hidden {
            display: none;
        }
        
        .blinking-suggestion-text {
            text-align: center;
            font-weight: bold;
            color: var(--secondary-color);
            animation: blink 1.5s infinite;
            margin: 10px 0;
        }
        
        @keyframes blink {
            0% { opacity: 1; }
            50% { opacity: 0.5; }
            100% { opacity: 1; }
        }
        
        .step-indicator {
            display: flex;
            justify-content: center;
            margin-bottom: 30px;
        }
        
        .step {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            background: #ddd;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 10px;
            font-weight: bold;
            color: #777;
        }
        
        .step.active {
            background: var(--primary-color);
            color: white;
        }
        
        .step.completed {
            background: var(--success-color);
            color: white;
        }
        
        .step-line {
            flex: 1;
            height: 2px;
            background: #ddd;
            margin: 0 5px;
            position: relative;
            top: 15px;
            max-width: 100px;
        }
        
        .step-line.completed {
            background: var(--success-color);
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="step-indicator">
            <div class="step completed">1</div>
            <div class="step-line completed"></div>
            <div class="step completed">2</div>
            <div class="step-line"></div>
            <div class="step active">3</div>
            <div class="step-line"></div>
            <div class="step">4</div>
        </div>

        <!-- STEP 3 -->
        <div id="step3-container">
            <h1>Step 3: Selected Package</h1>
            <p style="text-align:center; margin-bottom: 30px; font-size: 1.2rem; font-weight: bold; color: #e53935;">
                Choose your budget, Fill your website details & Submit in one click!
            </p>

            <div class="budget-container">
                <h2>Choose Your Budget</h2>
                <div class="price-display" id="price-display">$25</div>
                <div id="suggestion-text" class="blinking-suggestion-text" style="display:none;"></div>
                <div class="slider-wrapper">
                    <div class="animated-hand-text-container">
                        <div id="animated-text" class="animated-text">
                            <span class="hand-icon hand-left">👈</span> Slide to Change Package <span class="hand-icon hand-right">👉</span>
                        </div>
                    </div>
                    <input type="range" id="budget-slider" class="budget-slider" min="0" max="8" value="0">
                </div>
            </div>

            <div class="campaign-details">
                <h3 id="package-name-heading"></h3>
                <h4>Campaign Details</h4>
                <ul id="campaign-features">
                    <!-- Features will be populated by JavaScript -->
                </ul>
            </div>
            
            <form id="final-campaign-form" style="margin-top: 30px; padding: 20px; box-shadow: none;">
                <h4 style="text-align: center; color: #1a237e;">Order details</h4>
                <div class="form-group">
                    <label for="campaign-name">Campaign Name:</label>
                    <input type="text" id="campaign-name" name="campaign_name" placeholder="E.g., My first campaign" required>
                </div>
                <div class="form-group">
                    <label for="keywords">Target Keywords (separated by commas):</label>
                    <input type="text" id="keywords" name="keywords" placeholder="E.g., SEO tool, marketing software" required>
                </div>
                <div class="form-group">
                    <label for="website-urls">Website URLs (one per line):</label>
                    <textarea id="website-urls" name="website" rows="5" required
                        placeholder="http://example.com/page-one.html
http://example.com/page-two.html
http://example.com/page-three.html"></textarea>
                </div>
            </form>

            <div style="text-align:center; margin-top: 40px;">
                <button id="step3-prev" class="btn-secondary" type="button">Previous</button>
                <button id="step3-next" class="btn-primary" type="button">Campaign Scheduling</button>
            </div>
        </div>

        <!-- STEP 4 -->
        <div id="step4-container" class="hidden">
            <h1 id="step4-heading">Step 4: Campaign Scheduling</h1>

            <div class="campaign-details">
                <h3 id="step4-package-summary-heading"></h3>
            </div>

            <div class="budget-container" style="margin-top: 30px;">
                <h2>Campaign Scheduling Area</h2>
                <p style="text-align: center; font-weight: bold; margin-bottom: 20px;">
                    Please choose your preferred billing option below.
                </p>
                
                <div class="tab-container" id="billing-tabs">
                    <button class="tab-button active" data-period="onetime">Onetime Campaign</button>
                    <button class="tab-button" data-period="weekly">Weekly Campaign</button>
                    <button class="tab-button" data-period="monthly">Monthly Campaign</button>
                    <button class="tab-button" data-period="six-months">6 Months Campaign</button>
                </div>

                <div class="form-group">
                    <label for="coupon-code">Coupon Code:</label>
                    <div class="coupon-box">
                        <input type="text" id="coupon-code" name="coupon-code" placeholder="Enter coupon code">
                        <span id="coupon-validation-icon" class="validation-icon" style="display:none;">❓</span>
                        <span id="coupon-validation-msg" class="validation-msg" style="display:none;">Not validated</span>
                    </div>
                    <button id="apply-coupon-btn" type="button" class="coupon-btn">Apply Coupon</button>
                </div>
                
                <div class="form-group">
                    <label for="payment-gateway">Select Payment Gateway:</label>
                    <select id="payment-gateway" class="dropdown-select">
                        <option value="">Select a payment method</option>
                        <option value="paypal">PayPal</option>
                        <option value="stripe">Stripe</option>
                        <option value="bank_transfer">Bank Transfer</option>
                        <option value="crypto">Cryptocurrency</option>
                    </select>
                </div>

                <form id="final-submission-form" method="POST" action="process_campaign.php">
                    <input type="hidden" id="db-campaign-name" name="campaign_name">
                    <input type="hidden" id="db-website-urls" name="website_urls">
                    <input type="hidden" id="db-keywords" name="keywords">
                    <input type="hidden" id="db-package-id" name="package_id">
                    <input type="hidden" id="db-billing-period" name="billing_period" value="onetime">
                    <input type="hidden" id="db-amount" name="amount">
                    <input type="hidden" id="db-discount-amount" name="discount_amount" value="0">
                    <input type="hidden" id="db-payment-method" name="payment_method">

                    <div style="text-align:center; margin-top: 40px;">
                        <button id="step4-prev" class="btn-secondary" type="button">Previous</button>
                        <button id="pay-now-btn" class="btn-primary" type="submit">Pay Now</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Package data
        const packages = [
            { id: 1, Name: "Starter Package", price: "25", features: ["5 Backlinks", "Basic Reporting", "30-Day Support"] },
            { id: 2, Name: "Basic Package", price: "50", features: ["15 Backlinks", "Basic Reporting", "60-Day Support"] },
            { id: 3, Name: "Professional Package", price: "100", features: ["30 Backlinks", "Advanced Reporting", "90-Day Support", "Priority Submission"] },
            { id: 4, Name: "Business Package", price: "200", features: ["60 Backlinks", "Advanced Reporting", "120-Day Support", "Priority Submission", "Custom Strategy"] },
            { id: 5, Name: "Enterprise Package", price: "350", features: ["100 Backlinks", "Comprehensive Reporting", "180-Day Support", "Priority Submission", "Custom Strategy", "Dedicated Account Manager"] },
            { id: 6, Name: "Premium Package", price: "500", features: ["150 Backlinks", "Comprehensive Reporting", "365-Day Support", "Instant Submission", "Custom Strategy", "Dedicated Account Manager", "Competitor Analysis"] },
            { id: 7, Name: "Platinum Package", price: "750", features: ["250 Backlinks", "Comprehensive Reporting", "365-Day Support", "Instant Submission", "Custom Strategy", "Dedicated Account Manager", "Competitor Analysis", "Monthly Strategy Calls"] },
            { id: 8, Name: "Diamond Package", price: "1000", features: ["400 Backlinks", "Comprehensive Reporting", "365-Day Support", "Instant Submission", "Custom Strategy", "Dedicated Account Manager", "Competitor Analysis", "Weekly Strategy Calls", "White Label Reporting"] },
            { id: 9, Name: "Custom Package", price: "1500", features: ["Custom Backlinks", "Comprehensive Reporting", "365-Day Support", "Instant Submission", "Custom Strategy", "Dedicated Account Manager", "Competitor Analysis", "Weekly Strategy Calls", "White Label Reporting", "Custom Requirements"] }
        ];

        // DOM elements
        const step3Container = document.getElementById('step3-container');
        const step4Container = document.getElementById('step4-container');
        const budgetSlider = document.getElementById('budget-slider');
        const priceDisplay = document.getElementById('price-display');
        const packageNameHeading = document.getElementById('package-name-heading');
        const campaignFeatures = document.getElementById('campaign-features');
        const step4PackageSummary = document.getElementById('step4-package-summary-heading');
        const step3NextBtn = document.getElementById('step3-next');
        const step3PrevBtn = document.getElementById('step3-prev');
        const step4PrevBtn = document.getElementById('step4-prev');
        const billingTabs = document.querySelectorAll('.tab-button');
        const applyCouponBtn = document.getElementById('apply-coupon-btn');
        const couponInput = document.getElementById('coupon-code');
        const couponMsg = document.getElementById('coupon-validation-msg');
        const couponIcon = document.getElementById('coupon-validation-icon');
        const paymentGateway = document.getElementById('payment-gateway');
        const finalForm = document.getElementById('final-submission-form');

        // Initialize the page
        document.addEventListener('DOMContentLoaded', function() {
            updatePackageDetails(0);
            
            // Budget slider event
            budgetSlider.addEventListener('input', function() {
                updatePackageDetails(this.value);
            });
            
            // Step navigation events
            step3NextBtn.addEventListener('click', goToStep4);
            step3PrevBtn.addEventListener('click', goToPreviousStep);
            step4PrevBtn.addEventListener('click', goToPreviousStep);
            
            // Billing tabs events
            billingTabs.forEach(tab => {
                tab.addEventListener('click', function() {
                    billingTabs.forEach(t => t.classList.remove('active'));
                    this.classList.add('active');
                    document.getElementById('db-billing-period').value = this.dataset.period;
                });
            });
            
            // Coupon application
            applyCouponBtn.addEventListener('click', applyCoupon);
            
            // Payment gateway change
            paymentGateway.addEventListener('change', function() {
                document.getElementById('db-payment-method').value = this.value;
            });
            
            // Form submission
            finalForm.addEventListener('submit', prepareFormData);
        });

        // Update package details based on slider value
        function updatePackageDetails(sliderValue) {
            const packageInfo = packages[sliderValue];
            priceDisplay.textContent = '$' + packageInfo.price;
            packageNameHeading.textContent = `Selected Package: ${packageInfo.Name} - $${packageInfo.price}`;
            
            // Update features list
            campaignFeatures.innerHTML = '';
            packageInfo.features.forEach(feature => {
                const li = document.createElement('li');
                li.textContent = '✓ ' + feature;
                campaignFeatures.appendChild(li);
            });
            
            // Update hidden form fields
            document.getElementById('db-package-id').value = packageInfo.id;
            document.getElementById('db-amount').value = packageInfo.price;
        }

        // Navigate to step 4
        function goToStep4() {
            // Validate step 3 form
            const campaignName = document.getElementById('campaign-name').value;
            const keywords = document.getElementById('keywords').value;
            const websiteUrls = document.getElementById('website-urls').value;
            
            if (!campaignName || !keywords || !websiteUrls) {
                alert('Please fill all the required fields before proceeding.');
                return;
            }
            
            // Update step 4 summary
            const packageInfo = packages[budgetSlider.value];
            step4PackageSummary.innerHTML = `
                Selected Package: <span style="color:red; font-weight:bold;">${packageInfo.Name} - $${packageInfo.price}</span>
            `;
            
            // Show step 4, hide step 3
            step3Container.classList.add('hidden');
            step4Container.classList.remove('hidden');
            
            // Update step indicator
            updateStepIndicator(4);
        }

        // Go back to previous step
        function goToPreviousStep() {
            step4Container.classList.add('hidden');
            step3Container.classList.remove('hidden');
            
            // Update step indicator
            updateStepIndicator(3);
        }

        // Apply coupon code
        function applyCoupon() {
            const couponCode = couponInput.value.trim();
            
            if (!couponCode) {
                couponMsg.textContent = 'Please enter a coupon code';
                couponMsg.className = 'validation-msg validation-warning';
                couponMsg.style.display = 'inline-block';
                couponIcon.style.display = 'none';
                return;
            }
            
            // Simulate coupon validation (in a real app, this would be an API call)
            let discount = 0;
            let discountType = 'percent';
            let isValid = false;
            
            if (couponCode === 'WELCOME10') {
                discount = 10;
                isValid = true;
            } else if (couponCode === 'SAVE20') {
                discount = 20;
                isValid = true;
            } else if (couponCode === 'FREETRIAL') {
                discount = 100;
                isValid = true;
            }
            
            if (isValid) {
                const packageInfo = packages[budgetSlider.value];
                const basePrice = parseFloat(packageInfo.price);
                let discountAmount = 0;
                let finalPrice = basePrice;
                
                if (discountType === 'percent') {
                    discountAmount = (basePrice * discount) / 100;
                    finalPrice = basePrice - discountAmount;
                } else {
                    discountAmount = discount;
                    finalPrice = basePrice - discountAmount;
                }
                
                couponMsg.innerHTML = `
                    ✅ Coupon applied successfully!<br>
                    Discount: <strong style="color:green;">$${discountAmount.toFixed(2)}</strong><br>
                    New Total: <strong style="color:#d63384;">$${finalPrice.toFixed(2)}</strong>
                `;
                couponMsg.className = 'validation-msg validation-success';
                couponMsg.style.display = 'inline-block';
                couponIcon.style.display = 'none';
                
                // Update the summary
                step4PackageSummary.innerHTML = `
                    Selected Package: <span style="color:red; font-weight:bold;">${packageInfo.Name} - $${basePrice.toFixed(2)}</span><br>
                    <span style="color:green; font-weight:bold;">Discount Applied: -$${discountAmount.toFixed(2)}</span><br>
                    <span style="color:#007bff; font-weight:bold;">Final Total: $${finalPrice.toFixed(2)}</span>
                `;
                
                // Update hidden form fields
                document.getElementById('db-discount-amount').value = discountAmount.toFixed(2);
                document.getElementById('db-amount').value = finalPrice.toFixed(2);
            } else {
                couponMsg.textContent = '❌ Invalid or expired coupon code.';
                couponMsg.className = 'validation-msg validation-error';
                couponMsg.style.display = 'inline-block';
                couponIcon.style.display = 'none';
            }
        }

        // Prepare form data before submission
        function prepareFormData(e) {
            // Prevent default form submission for demo purposes
            e.preventDefault();
            
            // Collect all form data
            const formData = {
                campaign_name: document.getElementById('campaign-name').value,
                website_urls: document.getElementById('website-urls').value,
                keywords: document.getElementById('keywords').value,
                package_id: document.getElementById('db-package-id').value,
                billing_period: document.getElementById('db-billing-period').value,
                amount: document.getElementById('db-amount').value,
                discount_amount: document.getElementById('db-discount-amount').value,
                payment_method: document.getElementById('db-payment-method').value
            };
            
            // In a real application, you would send this data to your server
            console.log('Form data to be submitted:', formData);
            
            // Show success message (in a real app, this would be a redirect to payment gateway)
            alert('Form data prepared for database insertion! Check console for details.');
            
            // Here you would typically:
            // 1. Send the data to your server via fetch or form submission
            // 2. Redirect to payment gateway if needed
            // 3. Handle the response
        }

        // Update step indicator
        function updateStepIndicator(step) {
            const steps = document.querySelectorAll('.step');
            const lines = document.querySelectorAll('.step-line');
            
            steps.forEach((s, i) => {
                if (i + 1 < step) {
                    s.classList.add('completed');
                    s.classList.remove('active');
                } else if (i + 1 === step) {
                    s.classList.add('active');
                    s.classList.remove('completed');
                } else {
                    s.classList.remove('active', 'completed');
                }
            });
            
            lines.forEach((line, i) => {
                if (i + 1 < step - 1) {
                    line.classList.add('completed');
                } else {
                    line.classList.remove('completed');
                }
            });
        }
    </script>
</body>
</html>